/* The scripts should always contain at least 10 functions :
	ScriptID, 
	DisplaySourceName(), 
	SourceSuperClassID(), 
	SourceClassID(), 
	DisplayDestinationName() 
	DestinationSuperClassID(), 
	DestinationClassID(), 
	AboutText() and 
	DefaultConversion, Conversion taking a param.
	
This script will have all aiSSurface Surface parameters within a function group, and some lines will be commented-out if params are not represented in the source material
	
*/

/* Class IDs"
 as reported by $.material.classid as string

 Physical Material Source:		"#(1030429932L, 3735928833L)"
 Arnold Standard Destination:	"#(2121471519L, 1660373836L)"
 
*/

--Script ID that will append to destination
fn ScriptID = 
(
	"No Displacement"
)

--Return the name to be display for the source in the Scene Converter UI
fn DisplaySourceName =
(
	"Physical Material"
)

--Returns the source class of the super class id 
fn SourceSuperClassID =
(
	3072 --Material
)

--Returns the class id of the source class we use as a source for conversion
fn SourceClassID =
(
	#(0x3d6b1cecL, 0xdeadc001L) -- 1030429932L, 3735928833L as reported by $.material.classid as string for Physical Material
)

--Return the name to be display for the destination in the Scene Converter UI - must be in the order than the destination class ids
fn DisplayDestinationName =
(
	"Arnold Standard Surface"
)

--Returns the destination class of the super class id
fn DestinationSuperClassID =
(
	3072 --Material
)
--Returns the class id of the class we use as a destination for conversion
-- so we convert from SourceClassID() to DestinationClassID()
fn DestinationClassID =
(
	#(2121471519L, 1660373836L) --Arnold Standard Material
)

--Validate that the source provided match with one of the SourceClass
fn VerifySource source =
(
	local ret = false	
	local src = SourceClassID()
	
	if(source == undefined) do return false
	
	if(classof src[1] == Array) then
    (
		for cd in src do
		(
			if((cd as string) == (source.ClassID as string)) do
			(
				ret = true
			)
		)
	)
    else
	(
		if((src as string) == (source.ClassID as string)) do
		(
			ret = true
		)
	)
    ret 
)

--Validate that the destination provided match with one of the DestinationClass
fn VerifyDestination destination =
(
	local ret = false	
	local aiSSurface = DestinationClassID()

	if(destination == undefined) do return false
	
	if(classof aiSSurface[1] == Array) then
    (
		for cd in aiSSurface do
		(
			if((cd as string) == (destination.ClassID as string)) do
			(
				ret = true
			)
		)
	)
    else
	(
		if((aiSSurface as string) == (destination.ClassID as string)) do
		(
			ret = true
		)
	)
    ret 
)

--Returns some information about this conversion script
fn AboutText =
(
	"Converts a Physical material to an Arnold Standard Surface"
)

/* aiSSurface Surface Coating Group */
fn SetCoating aiSSurface source =
(
	aiSSurface.coat = source.coating
	aiSSurface.coat_shader = source.coat_map
	aiSSurface.coat_connected = source.coat_map_on
	
	aiSSurface.coat_color = source.coat_color
	aiSSurface.coat_color_shader = source.coat_color_map	
	aiSSurface.coat_color_connected = source.coat_color_map_on
	
	-- Is the roughness inverted? 
	local rough = source.coat_roughness
	if(source.coat_roughness_inv == true ) do
	(		
		rough = 1.0 - source.coat_roughness	
	)
	
	aiSSurface.coat_roughness = rough
	aiSSurface.coat_roughness_shader = source.coat_rough_map	
	aiSSurface.coat_roughness_connected = source.coat_rough_map_on
	
	aiSSurface.coat_ior = source.coat_ior
	-- aiSSurface.coat_ior_Shader = source.		
	-- aiSSurface.coat_ior_connected = source.
	
	-- aiSSurface.coat_normal = source.
	
	-- Normal takes original Bump through a Bump2D shader 
	if(source.coat_bump_map != undefined) do
	(
		local Bump2DMap = ai_bump2d name:"Coating Bump"
	
		Bump2DMap.bump_map_shader = source.coat_bump_map
		Bump2DMap.bump_map = 1.0
		Bump2DMap.bump_height = source.clearcoat_bump_map_amt / 5.0f
	
		aiSSurface.coat_normal_shader = Bump2DMap
	)

	aiSSurface.coat_normal_connected = source.coat_bump_map_on
	
	aiSSurface.coat_affect_color = source.coat_affect_color
	-- aiSSurface.coat_affect_color_shader = source.
	-- aiSSurface.coat_affect_color_connected = source.
	
	aiSSurface.coat_affect_roughness = source.coat_affect_roughness
	-- aiSSurface.coat_affect_roughness_shader = source.
	-- aiSSurface.coat_affect_roughness_connected = source.
)

/* aiSSurface Basic Params Group */
fn SetBaseColor aiSSurface source =
( 
	aiSSurface.base = source.base_weight
	aiSSurface.base_shader = source.base_weight_map
	aiSSurface.base_connected = source.base_weight_map_on
	
	aiSSurface.base_color = source.base_color
	aiSSurface.base_color_shader = source.base_color_map
	aiSSurface.base_color_connected = source.base_color_map_on
	
	aiSSurface.diffuse_roughness = source.diff_roughness
	aiSSurface.diffuse_roughness_shader = source.diff_rough_map
	aiSSurface.diffuse_roughness_connected = source.diff_rough_map_on
	
	--aiSSurface.caustics = source. 
	-- aiSSurface.indirect_diffuse = source.
)
	
/* aiSSurface Surface Reflections Group */
fn SetReflections aiSSurface source =
(
	aiSSurface.specular = source.reflectivity	
	aiSSurface.specular_shader = source.reflectivity_map
	aiSSurface.specular_connected = source.reflectivity_map_on
	
	aiSSurface.specular_color = source.refl_color
	aiSSurface.specular_color_shader = source.refl_color_map
	aiSSurface.specular_color_connected = source.refl_color_map_on
	
	-- Is the roughness inverted...
	if(source.roughness_inv == true ) then
	(		
		aiSSurface.specular_roughness = 1.0 - source.roughness
	) else
	(
		aiSSurface.specular_roughness = source.roughness		
	)
	
	-- Does not handle the invert of a map
	aiSSurface.specular_roughness_shader = source.roughness_map
	aiSSurface.specular_roughness_connected = source.roughness_map_on
	
	aiSSurface.metalness = source.metalness
	aiSSurface.metalness_shader = source.metalness_map
	aiSSurface.metalness_connected = source.metalness_map_on
	
    aiSSurface.specular_ior = source.trans_ior
	aiSSurface.specular_ior_shader = source.trans_ior_map
	aiSSurface.specular_ior_connected = source.trans_ior_map_on
)

/* aiSSurface Transparency Group */
fn SetTransparency aiSSurface source =
(
	aiSSurface.transmission = source.transparency
	aiSSurface.transmission_shader = source.transparency_map
	aiSSurface.transmission_connected = source.transparency_map_on
	
	-- Generic has no support for Trans_Color. 
	aiSSurface.transmission_color = source.trans_color
	aiSSurface.transmission_color_shader = source.trans_color_map
	aiSSurface.transmission_color_connected = source.trans_color_map_on
	
	--aiSSurface.transmission_extra_roughness  = source.
	--aiSSurface.transmission_extra_roughness_shader =  source.
	--aiSSurface.transmission_extra_roughness_connected = source.

	aiSSurface.transmission_depth = source.trans_depth
	
	aiSSurface.thin_walled = source.thin_walled
	
	-- aiSSurface.transmission_dispersion = source.
	-- aiSSurface.transmission_dispersion_shader = source.
	-- aiSSurface.transmission_scatter_anisotrophy = source.
	-- aiSSurface.transmission_scatter_anisotrophy_shader = source.
	-- aiSSurface.transmission_statter_anisotrophy_connected = source.

)

/* aiSSurface Sub-Surface Scattering */
fn SetSubSurfaceScattering aiSSurface source = 
(
	aiSSurface.subsurface = source.scattering
	aiSSurface.subsurface_shader = source.scattering_map
	aiSSurface.subsurface_connected = source.scattering_map_on
	
	aiSSurface.subsurface_color = source.sss_color
	aiSSurface.subsurface_color_shader = source.sss_color_map
	aiSSurface.subsurface_color_connected = source.sss_color_map_on

	local depth = source.sss_depth * 10;
	
	aiSSurface.subsurface_radius.r = source.sss_scatter_color.r * depth
	aiSSurface.subsurface_radius.g = source.sss_scatter_color.g * depth
	aiSSurface.subsurface_radius.b = source.sss_scatter_color.b * depth
	
	--aiSSurface.subsurface_radius_shader = source.
	
	aiSSurface.subsurface_scale = source.sss_scale
	aiSSurface.subsurface_scale_shader = source.sss_scale_map
	aiSSurface.subsurface_scale_connected = source.sss_scale_map_on	
)

/* aiSSurface Emission Controls */
fn SetEmissive aiSSurface source =
(
	aiSSurface.emission_color = source.emit_color
	
	aiSSurface.emission_color_shader = source.emit_color_map
	aiSSurface.emission_color_connected = source.emit_color_map_on
	
	aiSSurface.emission = source.emission
	aiSSurface.emission_shader = source.emission_map
	aiSSurface.emission_connected = source.emission_map_on
		
	-- This adds a Multiply shader, so ignore if this material doesn't use emission
	if(source.emission != 0.0) do
	(
		-- Use a Multiply shader on Emission Color and the Efective Illuminance color
		local MultiplyMap = ai_multiply name:"Emission Color"
		aiSSurface.emission_color_shader = MultiplyMap
		MultiplyMap.input1 = source.emission_color		
		MultiplyMap.input1_shader = source.emission_color_map
		
		-- TODO: Get scene Physical Scale from Tone Mapper
		local phys_scale = 3.1415926535 / 1500.0
		local self_illum_intensity = source.emit_luminance * phys_scale
		local myColor  = convertKelvinToRGB source.emit_kelvin self_illum_intensity
		
		MultiplyMap.input2 = myColor	
	)
)

fn SetAnisotrophy aiSSurface source =
(
	if(source.brdf_mode == on) then
	(
		aiSSurface.specular_ior = source.trans_ior
		aiSSurface.specular_ior_shader = source.trans_ior_map
		aiSSurface.specular_ior_connected = source.trans_ior_map_on
		
	) else
	(
		aiSSurface.specular_ior = 1 + source.brdf_low
	)
	
	aiSSurface.specular_anisotropy = 1.0 - source.anisotropy
	aiSSurface.specular_anisotropy_shader = source.anisotropy_map
	aiSSurface.specular_anisotropy_connected = source.anisotropy_map_on

	aiSSurface.specular_rotation = source.anisoangle
	aiSSurface.specular_rotation_shader = source.aniso_angle_map
	aiSSurface.specular_rotation_connected = source.aniso_angle_map_on	
)

fn SetSpecialMaps aiSSurface source =
(
	-- Normal takes original Bump map through a Bump2D shader 
	if(source.bump_map != undefined) do
	(
		local Bump2DMap = ai_bump2d name:"Bump"
	
		Bump2DMap.bump_map_shader = source.bump_map
		Bump2DMap.bump_map = 1.0
		Bump2DMap.bump_height = source.bump_map_amt / 5.0f
	
		aiSSurface.normal_shader = Bump2DMap
	)
	
	aiSSurface.normal_connected = source.bump_map_on
	
    -- Displacement map has to be added to an Arnold Properties modifer, and the shader added there.
	-- This can't be done from here, as we don't have the selected object. Add modifier manually.
    -- = source.displacement_map_on
    -- = source.displacement_map_amt
    -- = source.displacement_map
	
	-- aiSSurface.opacity = source.
	aiSSurface.opacity_shader = source.cutout_map
	aiSSurface.opacity_connected = source.cutout_map_on
)
	
--Internal function that does the parameter mapping
fn ConvertFromSourceToaiSSurface source aiSSurface =
(
	if ( false == VerifySource Source or 
		 false == VerifyDestination aiSSurface
		) do
	(
		--Not the suitable nodes
		return undefined
	)
   
	aiSSurface.name = source.name	
    	
	SetCoating aiSSurface source
	SetBaseColor aiSSurface source
	SetReflections aiSSurface source
	SetTransparency aiSSurface source
	SetSubSurfaceScattering aiSSurface source
	SetEmissive aiSSurface source 
	SetAnisotrophy aiSSurface source
	SetSpecialMaps aiSSurface source	

	aiSSurface
)

--This function is use as entry when the source is missing (due to a missing plugin) and cannot be completly loaded. 
--In that case a default object is created and returned.
fn DefaultConversion source /*Not used*/=
(
	--Create the aiSSurface material
	newMat = standard_surface()
	
	--return the new material
	newMat
)

--Main entry point from this script
--This function handles the material's creation
fn Conversion sourceMat =
(
	if (false == VerifySource sourceMat )  do
	(
		--Not the suitable node
		return undefined
	)	
	
	--Create the aiSSurface material
	newMat = ai_standard_surface()
	
	--Call the parameters mapping function to convert
	ConvertFromSourceToaiSSurface sourceMat newMat
	
	--return the new material
	newMat
)
